/*
 * Decompiled with CFR 0.152.
 */
package com.firebase.client.authentication;

import com.firebase.client.AuthData;
import com.firebase.client.CredentialStore;
import com.firebase.client.Firebase;
import com.firebase.client.FirebaseError;
import com.firebase.client.authentication.JsonBasicResponseHandler;
import com.firebase.client.authentication.RequestHandler;
import com.firebase.client.core.AuthExpirationBehavior;
import com.firebase.client.core.Context;
import com.firebase.client.core.Path;
import com.firebase.client.core.PersistentConnection;
import com.firebase.client.core.Repo;
import com.firebase.client.core.RepoInfo;
import com.firebase.client.utilities.HttpUtilities;
import com.firebase.client.utilities.LogWrapper;
import com.firebase.client.utilities.Utilities;
import com.firebase.client.utilities.encoding.JsonHelpers;
import com.shaded.fasterxml.jackson.core.JsonProcessingException;
import com.shaded.fasterxml.jackson.core.type.TypeReference;
import java.io.IOException;
import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Semaphore;
import org.shaded.apache.http.client.methods.HttpUriRequest;
import org.shaded.apache.http.impl.client.DefaultHttpClient;
import org.shaded.apache.http.params.BasicHttpParams;
import org.shaded.apache.http.params.HttpConnectionParams;

public class AuthenticationManager {
    private static final String TOKEN_KEY = "token";
    private static final String USER_DATA_KEY = "userData";
    private static final String AUTH_DATA_KEY = "authData";
    private static final String ERROR_KEY = "error";
    private static final String CUSTOM_PROVIDER = "custom";
    private static final String LOG_TAG = "AuthenticationManager";
    private static final int CONNECTION_TIMEOUT = 20000;
    private final Context context;
    private final Repo repo;
    private final RepoInfo repoInfo;
    private final PersistentConnection connection;
    private final CredentialStore store;
    private final LogWrapper logger;
    private final Set<Firebase.AuthStateListener> listenerSet;
    private AuthData authData;
    private AuthAttempt currentAuthAttempt;

    public AuthenticationManager(Context context, Repo repo, RepoInfo repoInfo, PersistentConnection connection) {
        this.context = context;
        this.repo = repo;
        this.repoInfo = repoInfo;
        this.connection = connection;
        this.authData = null;
        this.store = context.getCredentialStore();
        this.logger = context.getLogger(LOG_TAG);
        this.listenerSet = new HashSet<Firebase.AuthStateListener>();
    }

    private void fireEvent(Runnable r) {
        this.context.getEventTarget().postEvent(r);
    }

    private void fireOnSuccess(final Firebase.ValueResultHandler handler, final Object result) {
        if (handler != null) {
            this.fireEvent(new Runnable(){

                @Override
                public void run() {
                    handler.onSuccess(result);
                }
            });
        }
    }

    private void fireOnError(final Firebase.ValueResultHandler handler, final FirebaseError error) {
        if (handler != null) {
            this.fireEvent(new Runnable(){

                @Override
                public void run() {
                    handler.onError(error);
                }
            });
        }
    }

    private Firebase.ValueResultHandler ignoreResultValueHandler(final Firebase.ResultHandler handler) {
        return new Firebase.ValueResultHandler(){

            public void onSuccess(Object result) {
                handler.onSuccess();
            }

            @Override
            public void onError(FirebaseError error) {
                handler.onError(error);
            }
        };
    }

    private void preemptAnyExistingAttempts() {
        if (this.currentAuthAttempt != null) {
            FirebaseError error = new FirebaseError(-5, "Due to another authentication attempt, this authentication attempt was aborted before it could complete.");
            this.currentAuthAttempt.fireError(error);
            this.currentAuthAttempt = null;
        }
    }

    private FirebaseError decodeErrorResponse(Object errorResponse) {
        String code = Utilities.getOrNull(errorResponse, "code", String.class);
        String message = Utilities.getOrNull(errorResponse, "message", String.class);
        String details = Utilities.getOrNull(errorResponse, "details", String.class);
        if (code != null) {
            return FirebaseError.fromStatus(code, message, details);
        }
        String errorMessage = message == null ? "Error while authenticating." : message;
        return new FirebaseError(-999, errorMessage, details);
    }

    private boolean attemptHasBeenPreempted(AuthAttempt attempt) {
        return attempt != this.currentAuthAttempt;
    }

    private AuthAttempt newAuthAttempt(Firebase.AuthResultHandler handler) {
        this.preemptAnyExistingAttempts();
        this.currentAuthAttempt = new AuthAttempt(handler);
        return this.currentAuthAttempt;
    }

    private AuthAttempt newAuthAttempt(Firebase.AuthListener listener) {
        this.preemptAnyExistingAttempts();
        this.currentAuthAttempt = new AuthAttempt(listener);
        return this.currentAuthAttempt;
    }

    private void fireAuthErrorIfNotPreempted(final FirebaseError error, final AuthAttempt attempt) {
        if (!this.attemptHasBeenPreempted(attempt)) {
            if (attempt != null) {
                this.fireEvent(new Runnable(){

                    @Override
                    public void run() {
                        attempt.fireError(error);
                    }
                });
            }
            this.currentAuthAttempt = null;
        }
    }

    private void checkServerSettings() {
        if (this.repoInfo.isDemoHost()) {
            this.logger.warn("Firebase authentication is supported on production Firebases only (*.firebaseio.com). To secure your Firebase, create a production Firebase at https://www.firebase.com.");
        } else if (this.repoInfo.isCustomHost() && !this.context.isCustomAuthenticationServerSet()) {
            throw new IllegalStateException("For a custom firebase host you must first set your authentication server before using authentication features!");
        }
    }

    private String getFirebaseCredentialIdentifier() {
        return this.repoInfo.host;
    }

    private void scheduleNow(Runnable r) {
        this.context.getRunLoop().scheduleNow(r);
    }

    private AuthData parseAuthData(String token, Map<String, Object> rawAuthData, Map<String, Object> userData) {
        HashMap<String, Object> providerData;
        String provider;
        Object expiresObj;
        Map authData = Utilities.getOrNull(rawAuthData, "auth", Map.class);
        if (authData == null) {
            this.logger.warn("Received invalid auth data: " + rawAuthData);
        }
        long expires = (expiresObj = rawAuthData.get("expires")) == null ? 0L : (expiresObj instanceof Integer ? (long)((Integer)expiresObj).intValue() : (expiresObj instanceof Long ? (Long)expiresObj : (expiresObj instanceof Double ? ((Double)expiresObj).longValue() : 0L)));
        String uid = Utilities.getOrNull(authData, "uid", String.class);
        if (uid == null) {
            uid = Utilities.getOrNull(userData, "uid", String.class);
        }
        if ((provider = Utilities.getOrNull(authData, "provider", String.class)) == null) {
            provider = Utilities.getOrNull(userData, "provider", String.class);
        }
        if (provider == null) {
            provider = CUSTOM_PROVIDER;
        }
        if (uid == null || uid.isEmpty()) {
            this.logger.warn("Received invalid auth data: " + authData);
        }
        if ((providerData = Utilities.getOrNull(userData, provider, Map.class)) == null) {
            providerData = new HashMap<String, Object>();
        }
        return new AuthData(token, expires, uid, provider, authData, providerData);
    }

    private void handleBadAuthStatus(FirebaseError error, AuthAttempt attempt, boolean revoked) {
        boolean expiredToken;
        boolean bl = expiredToken = error.getCode() == -6;
        if (expiredToken && this.context.getAuthExpirationBehavior() == AuthExpirationBehavior.PAUSE_WRITES_UNTIL_REAUTH) {
            if (this.logger.logsDebug()) {
                this.logger.debug("Pausing writes due to expired token.");
            }
            this.connection.pauseWrites();
        } else if (this.connection.writesPaused()) {
            assert (this.context.getAuthExpirationBehavior() == AuthExpirationBehavior.PAUSE_WRITES_UNTIL_REAUTH);
            if (this.logger.logsDebug()) {
                this.logger.debug("Invalid auth while writes are paused; keeping existing session.");
            }
        } else {
            this.clearSession();
        }
        this.updateAuthState(null);
        if (attempt != null) {
            if (revoked) {
                attempt.fireRevoked(error);
            } else {
                attempt.fireError(error);
            }
        }
    }

    private void handleAuthSuccess(String credential, Map<String, Object> authDataMap, Map<String, Object> optionalUserData, boolean isNewSession, AuthAttempt attempt) {
        if (isNewSession && (authDataMap.get("auth") != null || authDataMap.get("expires") != null) && !this.saveSession(credential, authDataMap, optionalUserData)) {
            this.logger.warn("Failed to store credentials! Authentication will not be persistent!");
        }
        AuthData authData = this.parseAuthData(credential, authDataMap, optionalUserData);
        this.updateAuthState(authData);
        if (attempt != null) {
            attempt.fireSuccess(authData);
        }
        if (this.connection.writesPaused()) {
            if (this.logger.logsDebug()) {
                this.logger.debug("Unpausing writes after successful login.");
            }
            this.connection.unpauseWrites();
        }
    }

    public void resumeSession() {
        try {
            String credentialData = this.store.loadCredential(this.getFirebaseCredentialIdentifier(), this.context.getSessionPersistenceKey());
            if (credentialData != null) {
                Map credentials = (Map)JsonHelpers.getMapper().readValue(credentialData, (TypeReference)new TypeReference<Map<String, Object>>(){});
                final String tokenString = Utilities.getOrNull(credentials, TOKEN_KEY, String.class);
                final Map authDataObj = Utilities.getOrNull(credentials, AUTH_DATA_KEY, Map.class);
                final Map userData = Utilities.getOrNull(credentials, USER_DATA_KEY, Map.class);
                if (authDataObj != null) {
                    AuthData authData = this.parseAuthData(tokenString, authDataObj, userData);
                    this.updateAuthState(authData);
                    this.context.getRunLoop().scheduleNow(new Runnable(){

                        @Override
                        public void run() {
                            AuthenticationManager.this.connection.auth(tokenString, new Firebase.AuthListener(){

                                @Override
                                public void onAuthError(FirebaseError error) {
                                    AuthenticationManager.this.handleBadAuthStatus(error, null, false);
                                }

                                @Override
                                public void onAuthSuccess(Object authData) {
                                    AuthenticationManager.this.handleAuthSuccess(tokenString, authDataObj, userData, false, null);
                                }

                                @Override
                                public void onAuthRevoked(FirebaseError error) {
                                    AuthenticationManager.this.handleBadAuthStatus(error, null, true);
                                }
                            });
                        }
                    });
                }
            }
        }
        catch (IOException e) {
            this.logger.warn("Failed resuming authentication session!", e);
            this.clearSession();
        }
    }

    private boolean saveSession(String token, Map<String, Object> authData, Map<String, Object> userData) {
        String firebaseId = this.getFirebaseCredentialIdentifier();
        String sessionId = this.context.getSessionPersistenceKey();
        this.store.clearCredential(firebaseId, sessionId);
        HashMap<String, Object> sessionMap = new HashMap<String, Object>();
        sessionMap.put(TOKEN_KEY, token);
        sessionMap.put(AUTH_DATA_KEY, authData);
        sessionMap.put(USER_DATA_KEY, userData);
        try {
            if (this.logger.logsDebug()) {
                this.logger.debug("Storing credentials for Firebase \"" + firebaseId + "\" and session \"" + sessionId + "\".");
            }
            String credentialData = JsonHelpers.getMapper().writeValueAsString(sessionMap);
            return this.store.storeCredential(firebaseId, sessionId, credentialData);
        }
        catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private boolean clearSession() {
        String firebaseId = this.getFirebaseCredentialIdentifier();
        String sessionId = this.context.getSessionPersistenceKey();
        if (this.logger.logsDebug()) {
            this.logger.debug("Clearing credentials for Firebase \"" + firebaseId + "\" and session \"" + sessionId + "\".");
        }
        return this.store.clearCredential(firebaseId, sessionId);
    }

    private void updateAuthState(final AuthData authData) {
        boolean changed = this.authData == null ? authData != null : !this.authData.equals(authData);
        this.authData = authData;
        if (changed) {
            for (final Firebase.AuthStateListener listener : this.listenerSet) {
                this.fireEvent(new Runnable(){

                    @Override
                    public void run() {
                        listener.onAuthStateChanged(authData);
                    }
                });
            }
        }
    }

    private String buildUrlPath(String urlPath) {
        StringBuilder path = new StringBuilder();
        path.append("/v2/");
        path.append(this.repoInfo.namespace);
        if (!urlPath.startsWith("/")) {
            path.append("/");
        }
        path.append(urlPath);
        return path.toString();
    }

    private void makeRequest(String urlPath, HttpUtilities.HttpRequestType type, Map<String, String> urlParams, Map<String, String> requestParams, final RequestHandler handler) {
        HashMap<String, String> actualUrlParams = new HashMap<String, String>(urlParams);
        actualUrlParams.put("transport", "json");
        actualUrlParams.put("v", this.context.getPlatformVersion());
        final HttpUriRequest request = HttpUtilities.requestWithType(this.context.getAuthenticationServer(), this.buildUrlPath(urlPath), type, actualUrlParams, requestParams);
        if (this.logger.logsDebug()) {
            URI uri = request.getURI();
            String scheme = uri.getScheme();
            String authority = uri.getAuthority();
            String path = uri.getPath();
            int numQueryParams = uri.getQuery().split("&").length;
            this.logger.debug(String.format("Sending request to %s://%s%s with %d query params", scheme, authority, path, numQueryParams));
        }
        this.context.runBackgroundTask(new Runnable(){

            @Override
            public void run() {
                BasicHttpParams httpParameters = new BasicHttpParams();
                HttpConnectionParams.setConnectionTimeout(httpParameters, 20000);
                HttpConnectionParams.setSoTimeout(httpParameters, 20000);
                DefaultHttpClient httpClient = new DefaultHttpClient(httpParameters);
                httpClient.getParams().setParameter("http.protocol.cookie-policy", "compatibility");
                try {
                    final Map<String, Object> result = httpClient.execute(request, new JsonBasicResponseHandler());
                    if (result == null) {
                        throw new IOException("Authentication server did not respond with a valid response");
                    }
                    AuthenticationManager.this.scheduleNow(new Runnable(){

                        @Override
                        public void run() {
                            handler.onResult(result);
                        }
                    });
                }
                catch (IOException e) {
                    AuthenticationManager.this.scheduleNow(new Runnable(){

                        @Override
                        public void run() {
                            handler.onError(e);
                        }
                    });
                }
            }
        });
    }

    private void makeAuthenticationRequest(String urlPath, Map<String, String> params, Firebase.AuthResultHandler handler) {
        final AuthAttempt attempt = this.newAuthAttempt(handler);
        this.makeRequest(urlPath, HttpUtilities.HttpRequestType.GET, params, Collections.<String, String>emptyMap(), new RequestHandler(){

            @Override
            public void onResult(Map<String, Object> result) {
                Object errorResponse = result.get(AuthenticationManager.ERROR_KEY);
                String token = Utilities.getOrNull(result, AuthenticationManager.TOKEN_KEY, String.class);
                if (errorResponse == null && token != null) {
                    if (!AuthenticationManager.this.attemptHasBeenPreempted(attempt)) {
                        AuthenticationManager.this.authWithCredential(token, result, attempt);
                    }
                } else {
                    FirebaseError error = AuthenticationManager.this.decodeErrorResponse(errorResponse);
                    AuthenticationManager.this.fireAuthErrorIfNotPreempted(error, attempt);
                }
            }

            @Override
            public void onError(IOException e) {
                FirebaseError error = new FirebaseError(-24, "There was an exception while connecting to the authentication server: " + e.getLocalizedMessage());
                AuthenticationManager.this.fireAuthErrorIfNotPreempted(error, attempt);
            }
        });
    }

    private void makeOperationRequest(String urlPath, HttpUtilities.HttpRequestType type, Map<String, String> urlParams, Map<String, String> requestParams, Firebase.ResultHandler handler, boolean logUserOut) {
        this.makeOperationRequestWithResult(urlPath, type, urlParams, requestParams, this.ignoreResultValueHandler(handler), logUserOut);
    }

    private void makeOperationRequestWithResult(String urlPath, HttpUtilities.HttpRequestType type, Map<String, String> urlParams, Map<String, String> requestParams, final Firebase.ValueResultHandler<Map<String, Object>> handler, final boolean logUserOut) {
        this.makeRequest(urlPath, type, urlParams, requestParams, new RequestHandler(){

            @Override
            public void onResult(final Map<String, Object> result) {
                Object errorResponse = result.get(AuthenticationManager.ERROR_KEY);
                if (errorResponse == null) {
                    String uid;
                    if (logUserOut && (uid = Utilities.getOrNull(result, "uid", String.class)) != null && AuthenticationManager.this.authData != null && uid.equals(AuthenticationManager.this.authData.getUid())) {
                        AuthenticationManager.this.unauth(null, false);
                    }
                    AuthenticationManager.this.scheduleNow(new Runnable(){

                        @Override
                        public void run() {
                            AuthenticationManager.this.fireOnSuccess(handler, result);
                        }
                    });
                } else {
                    FirebaseError error = AuthenticationManager.this.decodeErrorResponse(errorResponse);
                    AuthenticationManager.this.fireOnError(handler, error);
                }
            }

            @Override
            public void onError(IOException e) {
                FirebaseError error = new FirebaseError(-24, "There was an exception while performing the request: " + e.getLocalizedMessage());
                AuthenticationManager.this.fireOnError(handler, error);
            }
        });
    }

    private void authWithCredential(final String credential, final Map<String, Object> optionalUserData, final AuthAttempt attempt) {
        if (attempt != this.currentAuthAttempt) {
            throw new IllegalStateException("Ooops. We messed up tracking which authentications are running!");
        }
        if (this.logger.logsDebug()) {
            this.logger.debug("Authenticating with credential of length " + credential.length());
        }
        this.currentAuthAttempt = null;
        this.connection.auth(credential, new Firebase.AuthListener(){

            @Override
            public void onAuthSuccess(Object authData) {
                AuthenticationManager.this.handleAuthSuccess(credential, (Map)authData, optionalUserData, true, attempt);
            }

            @Override
            public void onAuthRevoked(FirebaseError error) {
                AuthenticationManager.this.handleBadAuthStatus(error, attempt, true);
            }

            @Override
            public void onAuthError(FirebaseError error) {
                AuthenticationManager.this.handleBadAuthStatus(error, attempt, false);
            }
        });
    }

    public AuthData getAuth() {
        return this.authData;
    }

    public void unauth() {
        this.checkServerSettings();
        this.unauth(null);
    }

    public void unauth(Firebase.CompletionListener listener) {
        this.unauth(listener, true);
    }

    public void unauth(final Firebase.CompletionListener listener, boolean waitForCompletion) {
        this.checkServerSettings();
        final Semaphore semaphore = new Semaphore(0);
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                AuthenticationManager.this.preemptAnyExistingAttempts();
                AuthenticationManager.this.updateAuthState(null);
                semaphore.release();
                AuthenticationManager.this.clearSession();
                AuthenticationManager.this.connection.unauth(new Firebase.CompletionListener(){

                    @Override
                    public void onComplete(FirebaseError error, Firebase unusedRef) {
                        if (listener != null) {
                            Firebase ref = new Firebase(AuthenticationManager.this.repo, new Path(""));
                            listener.onComplete(error, ref);
                        }
                    }
                });
                if (AuthenticationManager.this.connection.writesPaused()) {
                    if (AuthenticationManager.this.logger.logsDebug()) {
                        AuthenticationManager.this.logger.debug("Unpausing writes after explicit unauth.");
                    }
                    AuthenticationManager.this.connection.unpauseWrites();
                }
            }
        });
        if (waitForCompletion) {
            try {
                semaphore.acquire();
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public void addAuthStateListener(final Firebase.AuthStateListener listener) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                AuthenticationManager.this.listenerSet.add(listener);
                final AuthData authData = AuthenticationManager.this.authData;
                AuthenticationManager.this.fireEvent(new Runnable(){

                    @Override
                    public void run() {
                        listener.onAuthStateChanged(authData);
                    }
                });
            }
        });
    }

    public void removeAuthStateListener(final Firebase.AuthStateListener listener) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                AuthenticationManager.this.listenerSet.remove(listener);
            }
        });
    }

    public void authAnonymously(final Firebase.AuthResultHandler handler) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                HashMap params = new HashMap();
                AuthenticationManager.this.makeAuthenticationRequest("/auth/anonymous", params, handler);
            }
        });
    }

    public void authWithPassword(final String email, final String password, final Firebase.AuthResultHandler handler) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                HashMap<String, String> params = new HashMap<String, String>();
                params.put("email", email);
                params.put("password", password);
                AuthenticationManager.this.makeAuthenticationRequest("/auth/password", params, handler);
            }
        });
    }

    public void authWithCustomToken(final String token, final Firebase.AuthResultHandler handler) {
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                AuthAttempt attempt = AuthenticationManager.this.newAuthAttempt(handler);
                AuthenticationManager.this.authWithCredential(token, null, attempt);
            }
        });
    }

    public void authWithFirebaseToken(final String token, final Firebase.AuthListener listener) {
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                AuthAttempt attempt = AuthenticationManager.this.newAuthAttempt(listener);
                AuthenticationManager.this.authWithCredential(token, null, attempt);
            }
        });
    }

    public void authWithOAuthToken(String provider, String token, Firebase.AuthResultHandler handler) {
        if (token == null) {
            throw new IllegalArgumentException("Token must not be null!");
        }
        HashMap<String, String> params = new HashMap<String, String>();
        params.put("access_token", token);
        this.authWithOAuthToken(provider, params, handler);
    }

    public void authWithOAuthToken(final String provider, final Map<String, String> params, final Firebase.AuthResultHandler handler) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                String url = String.format("/auth/%s/token", provider);
                AuthenticationManager.this.makeAuthenticationRequest(url, params, handler);
            }
        });
    }

    public void createUser(String email, String password, Firebase.ResultHandler handler) {
        this.createUser(email, password, this.ignoreResultValueHandler(handler));
    }

    public void createUser(final String email, final String password, final Firebase.ValueResultHandler<Map<String, Object>> handler) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                HashMap<String, String> requestParams = new HashMap<String, String>();
                requestParams.put("email", email);
                requestParams.put("password", password);
                AuthenticationManager.this.makeOperationRequestWithResult("/users", HttpUtilities.HttpRequestType.POST, Collections.emptyMap(), requestParams, handler, false);
            }
        });
    }

    public void removeUser(final String email, final String password, final Firebase.ResultHandler handler) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                HashMap<String, String> urlParams = new HashMap<String, String>();
                urlParams.put("password", password);
                String url = String.format("/users/%s", email);
                AuthenticationManager.this.makeOperationRequest(url, HttpUtilities.HttpRequestType.DELETE, urlParams, Collections.emptyMap(), handler, true);
            }
        });
    }

    public void changePassword(final String email, final String oldPassword, final String newPassword, final Firebase.ResultHandler handler) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                HashMap<String, String> urlParams = new HashMap<String, String>();
                urlParams.put("oldPassword", oldPassword);
                HashMap<String, String> requestParams = new HashMap<String, String>();
                requestParams.put("password", newPassword);
                String url = String.format("/users/%s/password", email);
                AuthenticationManager.this.makeOperationRequest(url, HttpUtilities.HttpRequestType.PUT, urlParams, requestParams, handler, false);
            }
        });
    }

    public void changeEmail(final String oldEmail, final String password, final String newEmail, final Firebase.ResultHandler handler) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                HashMap<String, String> urlParams = new HashMap<String, String>();
                urlParams.put("password", password);
                HashMap<String, String> requestParams = new HashMap<String, String>();
                requestParams.put("email", newEmail);
                String url = String.format("/users/%s/email", oldEmail);
                AuthenticationManager.this.makeOperationRequest(url, HttpUtilities.HttpRequestType.PUT, urlParams, requestParams, handler, false);
            }
        });
    }

    public void resetPassword(final String email, final Firebase.ResultHandler handler) {
        this.checkServerSettings();
        this.scheduleNow(new Runnable(){

            @Override
            public void run() {
                String url = String.format("/users/%s/password", email);
                Map params = Collections.emptyMap();
                AuthenticationManager.this.makeOperationRequest(url, HttpUtilities.HttpRequestType.POST, params, params, handler, false);
            }
        });
    }

    private class AuthAttempt {
        private Firebase.AuthResultHandler handler;
        private final Firebase.AuthListener legacyListener;

        AuthAttempt(Firebase.AuthResultHandler handler) {
            this.handler = handler;
            this.legacyListener = null;
        }

        AuthAttempt(Firebase.AuthListener legacyListener) {
            this.legacyListener = legacyListener;
            this.handler = null;
        }

        public void fireError(final FirebaseError error) {
            if (this.legacyListener != null || this.handler != null) {
                AuthenticationManager.this.fireEvent(new Runnable(){

                    @Override
                    public void run() {
                        if (AuthAttempt.this.legacyListener != null) {
                            AuthAttempt.this.legacyListener.onAuthError(error);
                        } else if (AuthAttempt.this.handler != null) {
                            AuthAttempt.this.handler.onAuthenticationError(error);
                            AuthAttempt.this.handler = null;
                        }
                    }
                });
            }
        }

        public void fireSuccess(final AuthData authData) {
            if (this.legacyListener != null || this.handler != null) {
                AuthenticationManager.this.fireEvent(new Runnable(){

                    @Override
                    public void run() {
                        if (AuthAttempt.this.legacyListener != null) {
                            AuthAttempt.this.legacyListener.onAuthSuccess(authData);
                        } else if (AuthAttempt.this.handler != null) {
                            AuthAttempt.this.handler.onAuthenticated(authData);
                            AuthAttempt.this.handler = null;
                        }
                    }
                });
            }
        }

        public void fireRevoked(final FirebaseError error) {
            if (this.legacyListener != null) {
                AuthenticationManager.this.fireEvent(new Runnable(){

                    @Override
                    public void run() {
                        AuthAttempt.this.legacyListener.onAuthRevoked(error);
                    }
                });
            }
        }
    }
}

